/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY                          *
 *                                                                            *
 * This program is free software; you can redistribute it and/or modify       *
 * it under the terms of the GNU General Public License as published by       *
 * the Free Software Foundation, either version 2 of the License, or (at      * 
 * your option) any later version.                                            *
 *                                                                            *
 * The ITX package is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY, without even the implied warranty of MERCHANTABILITY *
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
 * for more details.                                                          * 
 *                                                                            *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.                                                   *
 *                                                                            * 
 * Contact information:                                                       *
 * Donna Bergmark                                                             *
 * 484 Rhodes Hall                                                            *
 * Cornell University                                                         *
 * Ithaca, NY 14853-3801                                                      *
 * bergmark@cs.cornell.edu                                                    *
 ******************************************************************************/
import cnrg.itx.ds.*;
import java.util.*;
import java.net.*;
import java.io.*;

/**
 * This application class demonstrates how to use the directory service.
 */
class DirectoryClientTest
{
	/**
	 * The main entry point for the application. 
	 *
	 * @param args Array of parameters passed to the application
	 * via the command line.
	 */
	public static void main (String[] args)
	{
		System.out.println("Running ClientTest");
		DirectoryService ds;

		try {
			
			// Create an UserID and Password objects
			UserID id = new UserID("foo@cornell.edu");
			Password pw = new Password("foo");
		
			// Create a Location object
			String ipaddr = "123.23.23.23";
			String port = "4530";
			Location dummyloc = new Location(Location.INTERNET_TYPE, ipaddr + ":" + port, "dummy location");

			// DirectoryService default constructor will set up DSComm.dll to look for config file in the same directory as DSComm.dll resides
			// You can specify the directory path in which the config file resides by calling the following instantiation:
			//      ds = new DirectoryService("z:\\program files\\myapp\\resolv.conf");
			ds = new DirectoryService();

			// Authenticate the user -> this will fail because foo@cornell.edu is not registered in the ITX network
			ds.declareIdentity(id, pw);
			System.out.println("Succeed declareIdentity for : " + id.toString());

			// Register the current location of the client so that he/she can be reached
			ds.registerLocation(Location.ROAMING, dummyloc);
			System.out.println("Done register location as roaming.");

			// Change password -> this will fail because newpw is identical to the current pw
			Password newpw = new Password("foo");
			ds.setPIN(id, pw, newpw);
			
			// Change password -> this will fail because newid does not match current id
			UserID newid = new UserID("bar@cornell.edu");
			ds.setPIN(newid, pw, newpw);
				
			// Get the extension of the user
			Digits ext = ds.getExtension();
			
			// Get the default location(s) of the user
			Vector vec = ds.getDefaultLocationByExtension(ext);
			for (int i=0; i<vec.size(); i++) {
				Location loc = (Location)vec.elementAt(i);
				System.out.println("Type: " + loc.getType() + "  Value: " + loc.getValue() + "  Label: " + loc.getLabel());
			}

			// Get the custom location(s) of the user
			vec = ds.getCustomLocationByExtension(ext);
			for (int j=0; j<vec.size(); j++) {
				CustomLocationRecord rec = (CustomLocationRecord)vec.elementAt(j);
				CustomLocation loc = rec.getCustomLocation();
				System.out.println("ID: " + rec.getID() + "  Raw String: " + loc.toString());
			}
			
			// Get the current location list of another user -> this will fail because abc@cornell.edu is not registered in the ITX network
			UserID anotherUser = new UserID("abc@cornell.edu");
			LocationList loclist = ds.getLocationListByID(anotherUser);
			System.out.println("Size of location list : " + loclist.count());
			Location loc = loclist.first();
			while (loc!=null) {
				System.out.println("Type: " + loc.getType() + "   Value: " + loc.getValue() + "   Label: " + loc.getLabel());
				loc = loclist.next();
			}

			// Dump all the registered users in the ITX network
			System.out.println("-------------------DumpAllUsers------------------");
			Vector vec1 = ds.dumpAllUsers();
			for (int j=0; j< vec1.size(); j++) {
				UserProperty usr = (UserProperty)vec1.elementAt(j);
				System.out.println("User ID: " + usr.getUserID().toString());
				System.out.println("Extension: " + String.valueOf(usr.getExtension()));
				if (usr.getRoamingLocation() != null)
					System.out.println("Roam Loc: " + usr.getRoamingLocation().toString());
				else
					System.out.println("Roam Loc is null");
				System.out.println("Cust Msg: " + usr.getCustomMessage());					
				System.out.println();
			}

			// Unregister the current location of the client
			ds.unregisterLocation(Location.ROAMING, dummyloc);
			System.out.println("Done unregister roaming location.");

			System.out.println("FINISH");
		}catch (DirectoryServiceException ae) {
			ae.toString();
			ae.printStackTrace();
			System.out.println("Fail to initialize directory service : " + ae.toString());
			return;
		}
	}
}
